/*
 * Decompiled with CFR 0.152.
 */
package gov.loc.repository.bagit.conformance;

import gov.loc.repository.bagit.conformance.BagitWarning;
import gov.loc.repository.bagit.exceptions.InvalidBagitFileFormatException;
import gov.loc.repository.bagit.util.PathUtils;
import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.Normalizer;
import java.util.Collection;
import java.util.HashSet;
import java.util.ResourceBundle;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.helpers.MessageFormatter;

public final class ManifestChecker {
    private static final Logger logger = LoggerFactory.getLogger(ManifestChecker.class);
    private static final ResourceBundle messages = ResourceBundle.getBundle("MessageBundle");
    private static final String THUMBS_DB_FILE = "[Tt][Hh][Uu][Mm][Bb][Ss]\\.[Dd][Bb]";
    private static final String DS_STORE_FILE = "\\.[Dd][Ss]_[Ss][Tt][Oo][Rr][Ee]";
    private static final String SPOTLIGHT_FILE = "\\.[Ss][Pp][Oo][Tt][Ll][Ii][Gg][Hh][Tt]-[Vv]100";
    private static final String TRASHES_FILE = "\\.(_.)?[Tt][Rr][Aa][Ss][Hh][Ee][Ss]";
    private static final String FS_EVENTS_FILE = "\\.[Ff][Ss][Ee][Vv][Ee][Nn][Tt][Ss][Dd]";
    private static final String OS_FILES_REGEX = ".*data/([Tt][Hh][Uu][Mm][Bb][Ss]\\.[Dd][Bb]|\\.[Dd][Ss]_[Ss][Tt][Oo][Rr][Ee]|\\.[Ss][Pp][Oo][Tt][Ll][Ii][Gg][Hh][Tt]-[Vv]100|\\.(_.)?[Tt][Rr][Aa][Ss][Hh][Ee][Ss]|\\.[Ff][Ss][Ee][Vv][Ee][Nn][Tt][Ss][Dd])";

    private ManifestChecker() {
    }

    public static void checkManifests(Path bagitDir, Charset encoding, Set<BagitWarning> warnings, Collection<BagitWarning> warningsToIgnore) throws IOException, InvalidBagitFileFormatException {
        boolean missingTagManifest = true;
        try (DirectoryStream<Path> files = Files.newDirectoryStream(bagitDir);){
            for (Path file : files) {
                String filename = PathUtils.getFilename(file);
                if (!filename.contains("manifest-")) continue;
                if (filename.startsWith("manifest-")) {
                    ManifestChecker.checkData(file, encoding, warnings, warningsToIgnore, true);
                } else {
                    ManifestChecker.checkData(file, encoding, warnings, warningsToIgnore, false);
                    missingTagManifest = false;
                }
                String algorithm = filename.split("[-\\.]")[1];
                ManifestChecker.checkAlgorthm(algorithm, warnings, warningsToIgnore);
            }
        }
        if (!warningsToIgnore.contains((Object)BagitWarning.MISSING_TAG_MANIFEST) && missingTagManifest) {
            logger.warn(messages.getString("bag_missing_tag_manifest_warning"), (Object)bagitDir);
            warnings.add(BagitWarning.MISSING_TAG_MANIFEST);
        }
    }

    private static void checkData(Path manifestFile, Charset encoding, Set<BagitWarning> warnings, Collection<BagitWarning> warningsToIgnore, boolean isPayloadManifest) throws IOException, InvalidBagitFileFormatException {
        try (BufferedReader reader = Files.newBufferedReader(manifestFile, encoding);){
            HashSet<String> paths = new HashSet<String>();
            String line = reader.readLine();
            while (line != null) {
                String path = ManifestChecker.parsePath(line);
                path = ManifestChecker.checkForManifestCreatedWithMD5SumTools(path, warnings, warningsToIgnore);
                if (!warningsToIgnore.contains((Object)BagitWarning.DIFFERENT_CASE) && paths.contains(path.toLowerCase())) {
                    logger.warn(messages.getString("different_case_warning"), (Object)manifestFile, (Object)path);
                    warnings.add(BagitWarning.DIFFERENT_CASE);
                }
                paths.add(path.toLowerCase());
                if (encoding.name().startsWith("UTF")) {
                    ManifestChecker.checkNormalization(path, manifestFile.getParent(), warnings, warningsToIgnore);
                }
                ManifestChecker.checkForBagWithinBag(line, warnings, warningsToIgnore, isPayloadManifest);
                ManifestChecker.checkForRelativePaths(line, warnings, warningsToIgnore, manifestFile);
                ManifestChecker.checkForOSSpecificFiles(line, warnings, warningsToIgnore, manifestFile);
                line = reader.readLine();
            }
        }
    }

    static String parsePath(String line) throws InvalidBagitFileFormatException {
        String[] parts = line.split("\\s+", 2);
        if (parts.length < 2) {
            String formattedMessage = messages.getString("manifest_line_violated_spec_error");
            throw new InvalidBagitFileFormatException(MessageFormatter.format(formattedMessage, line).getMessage());
        }
        return parts[1];
    }

    private static String checkForManifestCreatedWithMD5SumTools(String path, Set<BagitWarning> warnings, Collection<BagitWarning> warningsToIgnore) {
        boolean startsWithStar;
        String fixedPath = path;
        boolean bl = startsWithStar = path.charAt(0) == '*';
        if (startsWithStar) {
            fixedPath = path.substring(1);
        }
        if (!warningsToIgnore.contains((Object)BagitWarning.MD5SUM_TOOL_GENERATED_MANIFEST) && startsWithStar) {
            logger.warn(messages.getString("md5sum_generated_line_warning"), (Object)path);
            warnings.add(BagitWarning.MD5SUM_TOOL_GENERATED_MANIFEST);
        }
        return fixedPath;
    }

    private static void checkNormalization(String path, Path rootDir, Set<BagitWarning> warnings, Collection<BagitWarning> warningsToIgnore) throws IOException {
        if (!warningsToIgnore.contains((Object)BagitWarning.DIFFERENT_NORMALIZATION)) {
            Path fileToCheck = rootDir.resolve(path).normalize();
            Path dirToCheck = fileToCheck.getParent();
            if (dirToCheck == null) {
                String formattedMessage = messages.getString("cannot_access_parent_path_error");
                throw new IOException(MessageFormatter.format(formattedMessage, fileToCheck).getMessage());
            }
            String normalizedFileToCheck = ManifestChecker.normalizePathToNFD(fileToCheck);
            try (DirectoryStream<Path> files = Files.newDirectoryStream(dirToCheck);){
                for (Path file : files) {
                    String normalizedFile = ManifestChecker.normalizePathToNFD(file);
                    if (file.equals(fileToCheck) || !normalizedFileToCheck.equals(normalizedFile)) continue;
                    logger.warn(messages.getString("different_normalization_in_manifest_warning"), (Object)fileToCheck);
                    warnings.add(BagitWarning.DIFFERENT_NORMALIZATION);
                }
            }
        }
    }

    static String normalizePathToNFD(Path path) {
        return Normalizer.normalize(path.toString(), Normalizer.Form.NFD);
    }

    private static void checkForBagWithinBag(String line, Set<BagitWarning> warnings, Collection<BagitWarning> warningsToIgnore, boolean isPayloadManifest) {
        if (!warningsToIgnore.contains((Object)BagitWarning.BAG_WITHIN_A_BAG) && isPayloadManifest && line.contains("manifest-")) {
            logger.warn(messages.getString("bag_within_bag_warning"));
            warnings.add(BagitWarning.BAG_WITHIN_A_BAG);
        }
    }

    private static void checkForRelativePaths(String line, Set<BagitWarning> warnings, Collection<BagitWarning> warningsToIgnore, Path manifestFile) {
        if (!warningsToIgnore.contains((Object)BagitWarning.LEADING_DOT_SLASH) && line.contains("./")) {
            logger.warn(messages.getString("leading_dot_slash_warning"), (Object)manifestFile, (Object)line);
            warnings.add(BagitWarning.LEADING_DOT_SLASH);
        }
    }

    private static void checkForOSSpecificFiles(String line, Set<BagitWarning> warnings, Collection<BagitWarning> warningsToIgnore, Path manifestFile) {
        if (!warningsToIgnore.contains((Object)BagitWarning.OS_SPECIFIC_FILES) && line.matches(OS_FILES_REGEX)) {
            logger.warn(messages.getString("os_specific_files_warning"), (Object)manifestFile, (Object)line);
            warnings.add(BagitWarning.OS_SPECIFIC_FILES);
        }
    }

    static void checkAlgorthm(String algorithm, Set<BagitWarning> warnings, Collection<BagitWarning> warningsToIgnore) {
        String upperCaseAlg = algorithm.toUpperCase();
        if (!warningsToIgnore.contains((Object)BagitWarning.WEAK_CHECKSUM_ALGORITHM) && (upperCaseAlg.startsWith("MD") || upperCaseAlg.matches("SHA(1|224|256|384)?"))) {
            logger.warn(messages.getString("weak_algorithm_warning"), (Object)algorithm);
            warnings.add(BagitWarning.WEAK_CHECKSUM_ALGORITHM);
        } else if (!warningsToIgnore.contains((Object)BagitWarning.NON_STANDARD_ALGORITHM) && !"SHA-512".equals(upperCaseAlg)) {
            logger.warn(messages.getString("non_standard_algorithm_warning"), (Object)algorithm);
            warnings.add(BagitWarning.NON_STANDARD_ALGORITHM);
        }
    }

    static String getOsFilesRegex() {
        return OS_FILES_REGEX;
    }
}

